/*****************************************************************************
*  LPC11Ux WS2811 LED driver demo LPC11uxx board
*
*  1. Use SCT1 timer to transmit WS2811 LED driver frames (24 bit)
*  2. SCT1_OUT0 is linked to P2_16
****************************************************************************/

#include "board.h"


void LEDDRIVER_open(void);
void LEDDRIVER_write(uint32_t rgb);                // write frame
void LEDDRIVER_haltAfterFrame(int on);             // (de)activate HALT after next frame
void LEDDRIVER_start(void);                        // start a transmission

#define DATA_OUT          0                               // use SCT1_OUT0 as data output
#define AUX_OUT           1                               // use SCT1_OUT1 as aux output
#define DATA_SPEED        400000

#define WS2811_FB         (SystemCoreClock/DATA_SPEED)    // full data bit time   = 90 (at 800 kb/s)
#define WS2811_T0H        ((WS2811_FB*20)/100)            // data bit 0 high time = 18 (at 800 kb/s)
#define WS2811_T1H        ((WS2811_FB*48)/100)            // data bit 1 high time ~ 43 (at 800 kb/s)
/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
volatile int busy;
volatile int count;

const uint32_t pattern[] =
{
    0x123456,                                      // 24 bit WS2811  frames
    0xFF00CC,
    0x555555,
    0x800001,
};
/*****************************************************************************
 * Private functions
 ****************************************************************************/
void SCT1_Init(void);
/*****************************************************************************
 * Public functions
 **********************************************************/

void LEDDRIVER_open(void)
{

	Chip_SCT_Init(LPC_SCT1);

	LPC_SCT1->STATE_L         = 2;                        				// start state
	LPC_SCT1->OUTPUT         &= ~((1 << DATA_OUT)          				// preset data output low
	                              |   (1 << AUX_OUT));     				// preset aux output low

	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_0, (WS2811_FB - 1));	// full bit period
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_1, (WS2811_T0H - 1));	// preset data output low
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_2, (WS2811_T1H - 1));	// preset aux output low

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_5, ENABLE_ALL_STATES);	 		// happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_5, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH0 	|	  // related to MATCH0_L
                                                                          SCT_COMBMODE_MATCH	|	  // match only
                                                                          SCT_STATELD_0			  |
                                                                          (31 << 15)			    ));  // add value to STATE



	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_4, 0x00000FFE);	 			// all data bit states except state 0
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2 	|	// related to MATCH2_L
                                                                          SCT_COMBMODE_MATCH	));	// match only

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_3, 0x00000FFF);	 			// all data bit states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_3, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_COMBMODE_MATCH	));	// match only

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_2, 0);	 						// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	  // related to MATCH1_L
                                                                          SCT_OUTSEL_H			  |	  // use OUTPUT for I/O condition
                                                                          (AUX_OUT << 6)		  |	  // Use AUX output signal
                                                                          SCT_IOCOND_LOW		  |	  // AUX = 0
                                                                          SCT_COMBMODE_AND		));	// match AND I/O

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_1, 0);	 						// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	  // related to MATCH1_L
                                                                          SCT_OUTSEL_H			  |	  // related to MATCH1_L
                                                                          (AUX_OUT << 6)		  |	  // use OUTPUT for I/O condition
                                                                          SCT_IOCOND_HIGH		  |	  // AUX = 0
                                                                          SCT_COMBMODE_AND		));	// match AND I/O

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_0,ENABLE_STATE0);	 			// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2 	|	  // related to MATCH2_L
                                                                          SCT_COMBMODE_MATCH	|	  // match only
                                                                          SCT_STATELD_1			  |	  // set STATE to a value
                                                                          SCT_STATEEV_6		   ));	// set STATE to 12




	    LPC_SCT1->OUT[AUX_OUT].SET  = (1 << 0);               // Event 0 toggles the AUX signal
	    LPC_SCT1->OUT[AUX_OUT].CLR  = (1 << 0);               // Event 0 toggles the AUX signal

	    LPC_SCT1->OUT[DATA_OUT].SET = (1 << 5)                // Event 5 sets the DATA signal
	                                   | (1 << 2)             // Event 2 sets the DATA signal
	                                   | (1 << 1);            // Event 1 sets the DATA signal
	    LPC_SCT1->OUT[DATA_OUT].CLR = (1 << 4)                // Event 4 clears the DATA signal
	                                   | (1 << 3)             // Event 3 clears the DATA signal
	                                   | (1 << 0);            // Event 0 clears the DATA signal


	    Chip_SCT_SetConflictResolution(LPC_SCT1, 0, DATA_OUT);	// on conflict: DATA signal doesn't change
	    Chip_SCT_SetConflictResolution(LPC_SCT1, 3, AUX_OUT);	// on conflict: AUX signal toggles

	    LPC_SCT1->LIMIT_L = (1 << 5);                         	// Event 5 limits the counter

	    LPC_SCT1->EVFLAG  = (1 << 0);				// clear event 0 irq flag


	    LPC_SCT1->EVEN   |= (1 << 0);				// Event 0 generates an irq

	    NVIC_EnableIRQ(SCT0_1_IRQn);       // enable SCT1 interrupt

}

void LEDDRIVER_write(uint32_t rgb)                         // function to write to a transmit buffer
{
    if (LPC_SCT1->OUTPUT & (1 << AUX_OUT))                 // aux output determines which buffer to write
    {
    	Chip_SCT_EventState(LPC_SCT1 , SCT_EVENT_2 , rgb & 0x3F);
    }
    else
    {
    	Chip_SCT_EventState(LPC_SCT1 , SCT_EVENT_1 , rgb & 0x3F);
    }
}

void LEDDRIVER_haltAfterFrame(int on)                      // (de)activate HALT after next frame
{
	 LPC_SCT1->HALT_L = (on << 0);                         // if last, event 10 halts the counter
}

void LEDDRIVER_start(void)                                 // start a frame transmission
{
	LPC_SCT1->COUNT_L = - WS2811_FB * 50;                  // guarantee 50 µs min time between transfers
    LPC_SCT1->STATE_L = 0;                                 // start state
    Chip_SCT_ClearControl(LPC_SCT1,SCT_CTRL_HALT_L);	   // unhalt it by clearing bit 2 of the CTRL register                         // start timer H
}



void SCT0_1_IRQHandler(void)
{
	int n;
	uint32_t bits;

	LPC_SCT1->EVFLAG = (1u << 0);       				// clear interrupt flag

    if(LPC_SCT1->STATE_L == 12)    						// for debug help
    	Chip_GPIO_SetPinToggle(LPC_GPIO, 2, 18);         // toggle pin P2_18 (LED1)


    if (++count < sizeof(pattern)/sizeof(pattern[0]))  // count nr of frames
    {
    	n = (count / 4) * 3;
    	bits = 0;
    	switch(count % 4)
    	{
    			case 0:
    	    		bits = pattern[n+0] & 0x3F;
    	    		break;
    	    	case 1:
    	    		bits = ((pattern[n+0] >> 6) & 0x03) | ((pattern[n+1] << 2) & 0x3C);
    	    		break;
    	    	case 2:
    	    		bits = ((pattern[n+1] >> 4) & 0x0F) | ((pattern[n+2] << 4) & 0x30);
    	    		break;
    	    	case 3:
    	    		bits = ((pattern[n+2] >> 2) & 0x3F);
    	    		break;
    	  }
    	     LEDDRIVER_write(bits);               // write next frame
    }
    else
    {
        LEDDRIVER_haltAfterFrame(1);                   // busy with last frame so halt after this one
    }

    if (LPC_SCT1->CTRL_L & 0x04)                       // if halted, transmission completed
    {
        busy = 0;
    }
}

int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();


	Chip_Clock_EnablePeriphClock( (CHIP_SYSCTL_CLOCK_T) ( SYSCTL_CLOCK_IOCON	  |				// enable IOCON clock
														  SYSCTL_CLOCK_GPIO ));				// enable SWM GPIO


	Chip_IOCON_PinMux(LPC_IOCON , 2, 16, IOCON_DIGMODE_EN, IOCON_FUNC1 );


	LEDDRIVER_open();

    while (1)                                              					// loop forever
    {
    	if (!busy)
    	{
    	    busy  = 1;                                 // transmission pending
    	    count = 0;                                 // used as frame counter
    	    LEDDRIVER_write(pattern[0]);               // preset first data word
    	    LEDDRIVER_haltAfterFrame(0);
    	    LEDDRIVER_start();                         // start transmission
    	 }
 //   	__WFI();
    }
}
